/*
  M5StickC (ESP32) - UART bridge + I2C tool on same pins (GPIO13 / GPIO25)

  Pines confirmados (externos, M5StickC):
    GPIO13  -> SDA (I2C)  y también UART RX (por defecto)
    GPIO25  -> SCL (I2C)  y también UART TX (por defecto)

  MODO UART (por defecto):
    - Puente USB Serial <-> Serial2 (anemómetro)
    - Lo que escribes en el PC se reenvía al anemómetro y viceversa.

  MODO I2C:
    - Se detiene UART, se liberan pines y se inicia I2C en los mismos GPIO.
    - Permite scan, read, write.
    - AUTO-REVERT: vuelve solo a UART si pasa el tiempo armado y no haces #ok.

  Comandos locales (NO se reenvían al anemómetro) -> empiezan por '#'
    #h               : ayuda
    #p               : estado actual
    #b115200         : cambiar baud UART anemómetro (9600/19200/38400/57600/115200)
    #swap            : intercambia RX/TX (solo si te hace falta; afecta también a I2C si lo dejas)
    #m uart          : volver a modo puente UART
    #m i2c           : pasar a modo I2C (arma auto-revert)
    #arm 15          : arma auto-revert a 15s (por defecto 15)
    #ok              : desarma auto-revert (te quedas en I2C sin timeout)
    #panic           : vuelve a UART inmediato

    I2C (en modo I2C):
    #scan            : escáner 0x08..0x77 (a 100k por defecto)
    #scan 400k       : escáner a 400k (o #scan 100k)
    #clk 100k        : fijar clock I2C (100k/400k)
    #addr 0x36       : fija dirección objetivo por defecto
    #rd 16           : lee 16 bytes desde addr por defecto
    #rd 0x36 16      : lee 16 bytes desde addr 0x36
    #wr 0x36 0x01 0x02 0x03  : escribe bytes a addr
    #reg 0x36 0x04 2 : escribe reg 0x04 y lee 2 bytes (mapa de registros típico)

  NOTA:
    - Para forzar cambio de modo del anemómetro a I2C: lo haces en MODO UART (puente),
      enviando el comando al anemómetro desde el PC; luego haces #m i2c aquí.
    - En este sketch, `setup()` fuerza PIN_A=13 y PIN_B=25. Si usas otros pines,
      cambia las constantes y el bloque de `setup()` para mantener coherencia.
*/

#include <Arduino.h>
#include <Wire.h>

// ------------------ Config pines (confirmados) ------------------
static int PIN_A = 33;  // SDA / UART RX (por defecto)
static int PIN_B = 32;  // SCL / UART TX (por defecto)

// UART anemómetro por Serial2
static int uart_rx = PIN_A;
static int uart_tx = PIN_B;
static uint32_t uart_baud = 115200;

// I2C en los mismos pines
static int i2c_sda = PIN_A;
static int i2c_scl = PIN_B;
static uint32_t i2c_clk_hz = 100000;

// Dirección objetivo opcional
static uint8_t i2c_addr_default = 0x36; // default anemómetro

// Auto-revert
static uint32_t arm_seconds = 30;
static bool i2c_armed = false;
static uint32_t i2c_deadline_ms = 0;


// Auto-switch I2C <-> UART por inactividad (solo si i2c_armed == true)
static uint32_t last_i2c_activity_ms = 0;  // última lectura I2C correcta (o entrada a I2C)
static uint32_t last_uart_activity_ms = 0; // último byte recibido por UART (o entrada a UART)
static bool lock_uart = false;             // forzar quedarse en UART (#m uart / #panic)

// Auto-lectura rápida (solo en I2C)
static uint32_t auto_interval_ms = 0;      // 0=OFF, 1000/100
static uint32_t auto_next_ms = 0;
// Modo
enum Mode { MODE_UART_BRIDGE, MODE_I2C_TOOL };

// ------------------ UI (menú) ------------------
enum UiMode { UI_MENU, UI_FRIENDLY, UI_TECH };
enum DetectedLink { DET_NONE, DET_UART, DET_I2C };

static UiMode ui_mode = UI_MENU;
static DetectedLink det_link = DET_NONE;
static uint8_t det_i2c_addr = 0x36;

enum MenuState {
  MS_NEED_DISCOVER,
  MS_CHOOSE_UI,
  MS_FRIENDLY_MAIN,
  MS_FRIENDLY_SET_MODE_PICK,
  MS_FRIENDLY_SET_MODE_CONFIRM,
  MS_FRIENDLY_ALARM_WIND,
  MS_FRIENDLY_ALARM_TIME_ON,
  MS_FRIENDLY_ALARM_HOLDOFF,
  MS_FRIENDLY_ALARM_PULSE,
  MS_FRIENDLY_ALARM_IDLELEVEL,
  MS_FRIENDLY_ALARM_CONFIRM
};
static MenuState menu_state = MS_NEED_DISCOVER;
static bool ui_choose_delay_pending = false; // delay de cortesía al mostrar el menú de interfaz
static bool ui_friendly_delay_pending = false; // delay antes de re-mostrar el menú del asistente

// variables temporales para el asistente (no guardadas hasta confirmar)
static int wiz_mode_choice = -1;     // 0..3 mapeado a l00
static long wiz_l01 = 0;             // km/h entero
static long wiz_l02 = 0;             // s
static long wiz_l03 = 0;             // s (anti-reentrada)
static long wiz_l04 = 1;             // s (1..32768)
static long wiz_l05 = 0;             // 0 reposo 0V, 1 reposo 3.3V (según FW)
static Mode mode = MODE_UART_BRIDGE;
static uint32_t hz = 100000;

// Entrada por USB
static String line;

// ------------------ Helpers parse ------------------
static bool parseUInt(const String &s, uint32_t &out) {
  String t = s;
  t.trim();
  if (t.length() == 0) return false;

  const char *c = t.c_str();
  char *endp = nullptr;
  unsigned long v = strtoul(c, &endp, 0); // base 0 -> admite 0x..
  if (endp == c) return false;
  out = (uint32_t)v;
  return true;
}

static void splitTokens(const String &s, String *tokens, int &ntok, int maxTok) {
  ntok = 0;
  int i = 0;
  while (i < (int)s.length() && ntok < maxTok) {
    while (i < (int)s.length() && isspace((unsigned char)s[i])) i++;
    if (i >= (int)s.length()) break;
    int j = i;
    while (j < (int)s.length() && !isspace((unsigned char)s[j])) j++;
    tokens[ntok++] = s.substring(i, j);
    i = j;
  }
}

// ------------------ UART/I2C transitions ------------------
// Pre-acondicionar pines compartidos (I2C/UART) para evitar flotacion / idle incorrecto tras reset
static void prepPinsForUart(int rxPin, int txPin) {
  // UART idle = HIGH. Si los pines quedan flotando (o en OD por I2C), el SA01 puede no reconocer el primer comando.
  pinMode(txPin, OUTPUT);
  digitalWrite(txPin, HIGH);
  pinMode(rxPin, INPUT_PULLUP);
  delay(5);
}

void startUart() {
  Serial2.end();
  delay(20);
  prepPinsForUart(uart_rx, uart_tx);
  Serial2.begin(uart_baud, SERIAL_8N1, uart_rx, uart_tx);
}

// Re-enganche robusto de UART (soluciona casos donde Serial2 no queda mapeado al arrancar)
static void uartKickInit() {
  // Guardar estado actual de pines (por si el usuario hizo #swap)
  int a0 = PIN_A, b0 = PIN_B;
  int rx0 = uart_rx, tx0 = uart_tx;

  // 1) Arranque normal con el mapeo actual
  startUart();
  delay(30);

  // 2) "Kick": iniciar con pines cruzados y volver (equivale al efecto swap-swap sin dejarlo permanente)
  PIN_A = b0; PIN_B = a0;
  uart_rx = PIN_A; uart_tx = PIN_B;
  i2c_sda = PIN_A; i2c_scl = PIN_B;
  startUart();
  delay(30);

  // 3) Restaurar mapeo original
  PIN_A = a0; PIN_B = b0;
  uart_rx = rx0; uart_tx = tx0;
  i2c_sda = PIN_A; i2c_scl = PIN_B;
  startUart();
  delay(30);
}

static void stopUartReleasePins() {
  Serial2.flush();
  Serial2.end();
  delay(20);
  pinMode(uart_rx, INPUT); // Hi-Z
  pinMode(uart_tx, INPUT); // Hi-Z
}

static void enterI2C(bool armTimeout, bool verbose=true) {
  stopUartReleasePins();

  // I2C usa los mismos pines (por diseño aquí)
  i2c_sda = PIN_A;
  i2c_scl = PIN_B;

  // En ESP32, re-llamar Wire.begin(...) no siempre reaplica la frecuencia.
  // Inicializamos con pines y fijamos el clock explícitamente.
  Wire.begin((int)i2c_sda, (int)i2c_scl);
  Wire.setClock((uint32_t)i2c_clk_hz);
  delay(10);

  mode = MODE_I2C_TOOL;

  // Entrada a I2C: inicia contador de inactividad
  last_i2c_activity_ms = millis();
  i2c_deadline_ms = last_i2c_activity_ms + arm_seconds * 1000UL;
  lock_uart = false; // al entrar a I2C, se libera lock UART (si lo hubiera)

  if (armTimeout) {
    i2c_armed = true;
    i2c_deadline_ms = millis() + arm_seconds * 1000UL;
  } else {
    i2c_armed = false;
  }

  if (verbose) {
  Serial.println();
  Serial.println(F("== MODE I2C TOOL =="));
  Serial.print(F("I2C SDA=")); Serial.print(i2c_sda);
  Serial.print(F(" SCL=")); Serial.print(i2c_scl);
  Serial.print(F(" CLK=")); Serial.print(i2c_clk_hz);
  Serial.print(F("  auto-revert="));
  if (i2c_armed) { Serial.print(F("ON (")); Serial.print(arm_seconds); Serial.println(F("s)")); }
  else Serial.println(F("OFF"));
  Serial.println(F("Usa: #scan  | #addr 0x.. | #rd N | #reg addr reg n | #panic"));
  }

}

static void exitI2CToUart(bool verbose=true) {
  // No existe Wire.end() en tu core; simplemente dejamos de usarlo
  delay(5);

  // Restaurar UART
  uart_rx = PIN_A;
  uart_tx = PIN_B;
  startUart();

  mode = MODE_UART_BRIDGE;

  // Entrada a UART: inicia contador de inactividad
  last_uart_activity_ms = millis();
  i2c_deadline_ms = last_uart_activity_ms + arm_seconds * 1000UL;


  if (verbose) {
  Serial.println();
  Serial.println(F("== MODE UART BRIDGE =="));
  }

}

// ------------------ Output helpers ------------------
static void printHelp() {
  Serial.println();
  Serial.println(F("=== M5StickC: UART bridge + I2C tool (GPIO13/25) ==="));
  Serial.println(F("Comandos locales (no se reenvian) empiezan por '#':"));
  Serial.println(F("  #h                 ayuda"));
  Serial.println(F("  #p                 estado"));
  Serial.println(F("  #b115200           baud UART (9600/19200/38400/57600/115200)"));
  Serial.println(F("  #swap              intercambia RX/TX (si te equivocas de cableado)"));
  Serial.println(F("  #m uart            volver a modo puente UART"));
  Serial.println(F("  #m i2c             pasar a modo I2C (arma auto-revert)"));
  Serial.println(F("  #arm 15            set auto-revert (segundos), default 15"));
  Serial.println(F("  #ok                desarma auto-revert (solo en I2C)"));
  Serial.println(F("  #panic             vuelve a UART inmediato"));
  Serial.println();
  Serial.println(F("I2C (solo en modo I2C):"));
  Serial.println(F("  #scan [100k|400k]  escaner 0x08..0x77"));
  Serial.println(F("  #clk 100k|400k     fija clock I2C"));
  Serial.println(F("  #addr 0x36         fija addr por defecto"));
  Serial.println(F("  #rd N              lee N bytes (addr por defecto)"));
  Serial.println(F("  #rd 0x36 N         lee N bytes desde addr"));
  Serial.println(F("  #wr 0x36 b1 b2..   escribe bytes (hex/dec)"));
  Serial.println(F("  #reg 0x36 0x04 N   write reg (1 byte) + read N bytes"));
  Serial.println();
  Serial.println(F("Atajos rapidos (solo en modo I2C, sin #):"));
  Serial.println(F("  1   leer 1 vez (0x36, 4 bytes) -> viento km/h + temp C"));
  Serial.println(F("  2   auto cada 1s (toggle)"));
  Serial.println(F("  3   auto cada 100ms (toggle)"));
  Serial.println(F("  4..7 I2C clk (toggle): 4=400kHz 5=500kHz 6=600kHz 7=650kHz (OFF->100kHz)"));
  Serial.println();
}

static void printStatus() {
  Serial.println();
  Serial.print(F("MODE=")); Serial.println(mode == MODE_UART_BRIDGE ? F("UART_BRIDGE") : F("I2C_TOOL"));
  Serial.print(F("PINS: A=")); Serial.print(PIN_A); Serial.print(F(" B=")); Serial.println(PIN_B);

  Serial.print(F("UART: RX=")); Serial.print(uart_rx);
  Serial.print(F(" TX=")); Serial.print(uart_tx);
  Serial.print(F(" BAUD=")); Serial.println(uart_baud);

  Serial.print(F("I2C : SDA=")); Serial.print(i2c_sda);
  Serial.print(F(" SCL=")); Serial.print(i2c_scl);
  Serial.print(F(" CLK=")); Serial.println(i2c_clk_hz);

  Serial.print(F("I2C addr default: "));
  if (i2c_addr_default == 0x00) Serial.println(F("(none)"));
  else { Serial.print(F("0x")); if (i2c_addr_default < 16) Serial.print('0'); Serial.println(i2c_addr_default, HEX); }

  Serial.print(F("Auto-switch: "));
  if (i2c_armed) {
    Serial.print(F("ON, remaining ms="));
    uint32_t now = millis();
    uint32_t remain = 0;
    if (mode == MODE_I2C_TOOL) {
      remain = (uint32_t)((now < i2c_deadline_ms) ? (i2c_deadline_ms - now) : 0);
    } else {
      if (!lock_uart) remain = (uint32_t)((now < i2c_deadline_ms) ? (i2c_deadline_ms - now) : 0);
      else remain = 0;
    }

    Serial.println(remain);
    Serial.print(F("Lock UART="));  Serial.println(lock_uart ? F("ON") : F("OFF"));
  } else {
    Serial.println(F("OFF"));
  }
}

// ------------------ I2C ops ------------------

static bool i2cQuickReadWindTemp(uint8_t addr, float &wind_kmh, float &temp_c, uint16_t &rawWind, uint16_t &rawTemp, uint8_t &got_bytes) {
  // Lectura fija de 4 bytes:
  //  [0..1] viento MSB/LSB (km/h *10)
  //  [2..3] temp   MSB/LSB (raw/100 - 40)
  Wire.requestFrom((int)addr, 4);
  uint8_t b[4];
  uint8_t n = 0;
  while (Wire.available() && n < 4) b[n++] = (uint8_t)Wire.read();
  got_bytes = n;
  if (n != 4) return false;

  rawWind = (uint16_t)(((uint16_t)b[0] << 8) | b[1]);
  rawTemp = (uint16_t)(((uint16_t)b[2] << 8) | b[3]);

  wind_kmh = ((float)rawWind) / 10.0f;
  temp_c = (((float)rawTemp) / 100.0f) - 40.0f;
  return true;
}

static void i2cQuickReadAndPrint(uint8_t addr) {
  float w_kmh, t_c;
  uint16_t rw, rt;

  uint8_t got = 0;

  bool ok = i2cQuickReadWindTemp(addr, w_kmh, t_c, rw, rt, got);
  if (!ok) {
    Serial.print(F("I2C read FAIL addr=0x"));
    if (addr < 16) Serial.print('0');
    Serial.print(addr, HEX);
    Serial.print(F(" got=")); Serial.println(got);
    return;
  }

  // Actividad I2C OK
  last_i2c_activity_ms = millis();
  i2c_deadline_ms = last_i2c_activity_ms + arm_seconds * 1000UL;

  Serial.print(F("W=")); Serial.print(w_kmh, 1); Serial.print(F(" km/h"));
  Serial.print(F("  T=")); Serial.print(t_c, 2); Serial.print(F(" C"));
  Serial.print(F("  RAW[W]=")); Serial.print(rw);
  Serial.print(F(" RAW[T]=")); Serial.println(rt);
}
static void i2cScan() {
  Serial.println();
  Serial.print(F("I2C scan (CLK=")); Serial.print(i2c_clk_hz); Serial.println(F(")"));
  uint8_t found = 0;

  for (uint8_t addr = 0x08; addr <= 0x77; addr++) {
    Wire.beginTransmission(addr);
    uint8_t err = Wire.endTransmission();
    if (err == 0) {
      Serial.print(F(" - Found 0x"));
      if (addr < 16) Serial.print('0');
      Serial.println(addr, HEX);
      found++;
    }
  }
  if (!found) Serial.println(F(" - No devices found"));
  Serial.println();
}

static void i2cReadBytes(uint8_t addr, uint16_t n) {
  Serial.println();
  Serial.print(F("I2C read addr=0x"));
  if (addr < 16) Serial.print('0');
  Serial.print(addr, HEX);
  Serial.print(F(" n=")); Serial.println(n);

  uint16_t got = 0;
  Wire.requestFrom((int)addr, (int)n, (int)true);

  while (Wire.available()) {
    uint8_t b = Wire.read();
    if ((got % 16) == 0) {
      Serial.print(F("\n  "));
      if (got < 16) Serial.print('0');
      Serial.print(got, HEX);
      Serial.print(F(": "));
    }
    if (b < 16) Serial.print('0');
    Serial.print(b, HEX);
    Serial.print(' ');
    got++;
  }

  Serial.print(F("\n\nGot ")); Serial.print(got); Serial.println(F(" bytes"));
  Serial.println();
}

static void i2cWriteBytes(uint8_t addr, const uint8_t *buf, uint16_t n) {
  Serial.println();
  Serial.print(F("I2C write addr=0x"));
  if (addr < 16) Serial.print('0');
  Serial.print(addr, HEX);
  Serial.print(F(" n=")); Serial.println(n);

  Wire.beginTransmission(addr);
  for (uint16_t i = 0; i < n; i++) Wire.write(buf[i]);
  uint8_t err = Wire.endTransmission();

  Serial.print(F("endTransmission err=")); Serial.println(err);
  Serial.println();
}

static void i2cRegRead(uint8_t addr, uint8_t reg, uint16_t n) {
  Serial.println();
  Serial.print(F("I2C reg-read addr=0x"));
  if (addr < 16) Serial.print('0');
  Serial.print(addr, HEX);
  Serial.print(F(" reg=0x"));
  if (reg < 16) Serial.print('0');
  Serial.print(reg, HEX);
  Serial.print(F(" n=")); Serial.println(n);

  Wire.beginTransmission(addr);
  Wire.write(reg);
  uint8_t err = Wire.endTransmission(false); // repeated start
  Serial.print(F("endTx(restart) err=")); Serial.println(err);

  if (err != 0) {
    Serial.println(F("Abort: no ACK / error"));
    Serial.println();
    return;
  }

  i2cReadBytes(addr, n);
}

// ------------------ Command handler ------------------
static bool setBaud(uint32_t b) {
  switch (b) {
    case 9600:
    case 19200:
    case 38400:
    case 57600:
    case 115200:
      uart_baud = b;
      if (mode == MODE_UART_BRIDGE) startUart();
      return true;
    default:
      return false;
  }
}

static bool setClk(const String &tok) {
  String t = tok;
  t.toLowerCase();
  if (t == "100k" || t == "100000") { i2c_clk_hz = 100000; return true; }
  if (t == "400k" || t == "400000") { i2c_clk_hz = 400000; return true; }
  uint32_t v;
  if (parseUInt(tok, v) && (v >= 10000 && v <= 1000000)) { i2c_clk_hz = v; return true; }
  return false;
}

static void uiPrintDiscoverMenu();
static void uiPrintChooseUiMenu();
static void handleLocalCommand(const String &cmdLine) {
  String cmd = cmdLine;
  cmd.trim();
  if (cmd == "#ui" || cmd == "#menu") {
    ui_mode = UI_MENU;
    if (det_link == DET_NONE) menu_state = MS_NEED_DISCOVER; else menu_state = MS_CHOOSE_UI;
    if (menu_state == MS_NEED_DISCOVER) uiPrintDiscoverMenu(); else uiPrintChooseUiMenu();
    return;
  }
  if (cmd == "#h") { printHelp(); return; }
  if (cmd == "#p") { printStatus(); return; }
  if (cmd == "#panic") { lock_uart = true; exitI2CToUart(); return; }
  if (cmd == "#ok") {
    if (mode == MODE_I2C_TOOL) {
      i2c_armed = false;
      Serial.println(F("OK: auto-switch OFF"));
    }
    return;
  }

  // Tokenizar
  String toks[16];
  int nt = 0;
  splitTokens(cmd, toks, nt, 16);
  if (nt == 0) return;

  // #bXXXX
  if (toks[0].startsWith("#b")) {
    uint32_t b = (uint32_t)toks[0].substring(2).toInt();
    if (setBaud(b)) Serial.println(F("OK baud set"));
    else Serial.println(F("BAUD no soportado (9600/19200/38400/57600/115200)"));
    return;
  }

  if (toks[0] == "#swap") {
    // Intercambia A<->B y por tanto RX/TX e I2C SDA/SCL (ojo: úsalo solo si lo necesitas)
    int tmp = PIN_A; PIN_A = PIN_B; PIN_B = tmp;
    uart_rx = PIN_A; uart_tx = PIN_B;
    i2c_sda = PIN_A; i2c_scl = PIN_B;

    if (mode == MODE_UART_BRIDGE) startUart();
    Serial.print(F("OK swap: A=")); Serial.print(PIN_A);
    Serial.print(F(" B=")); Serial.println(PIN_B);
    return;
  }

  if (toks[0] == "#arm") {
    if (nt >= 2) {
      uint32_t s;
      if (parseUInt(toks[1], s) && s >= 3 && s <= 120) {
        arm_seconds = s;
        Serial.print(F("OK arm_seconds=")); Serial.println(arm_seconds);
        if (i2c_armed) {
          uint32_t now = millis();
          if (mode == MODE_I2C_TOOL) i2c_deadline_ms = last_i2c_activity_ms + arm_seconds * 1000UL;
          else i2c_deadline_ms = last_uart_activity_ms + arm_seconds * 1000UL;
          (void)now;
          Serial.println(F("Re-armed deadline"));
        }
      } else {
        Serial.println(F("Uso: #arm <seg> (3..120)"));
      }
    } else {
      Serial.println(F("Uso: #arm <seg> (3..120)"));
    }
    return;
  }

  if (toks[0] == "#m") {
    if (nt < 2) { Serial.println(F("Uso: #m uart | #m i2c")); return; }
    String m = toks[1]; m.toLowerCase();

    if (m == "uart") {
      lock_uart = true;
      auto_interval_ms = 0;
      auto_next_ms = 0;
      if (mode != MODE_UART_BRIDGE) exitI2CToUart();
      else {
        // si ya estabas en UART, reinicia contador de inactividad
        last_uart_activity_ms = millis();
        i2c_deadline_ms = last_uart_activity_ms + arm_seconds * 1000UL;
        Serial.println(F("Ya estas en UART"));
      }
      return;
    }
    if (m == "i2c") {
      // #m i2c: entra en I2C y deja auto-switch habilitado
      i2c_armed = true;
      lock_uart = false;
      if (mode != MODE_I2C_TOOL) enterI2C(true);
      else {
        last_i2c_activity_ms = millis();
        i2c_deadline_ms = last_i2c_activity_ms + arm_seconds * 1000UL;
        Serial.println(F("Ya estas en I2C"));
      }
      return;
    }
    Serial.println(F("Uso: #m uart | #m i2c"));
    return;
  }

  // A partir de aquí, comandos solo válidos en I2C
  if (mode != MODE_I2C_TOOL) {
    Serial.println(F("Comando I2C solo disponible en modo I2C. Usa: #m i2c"));
    return;
  }

  if (toks[0] == "#clk") {
    if (nt >= 2 && setClk(toks[1])) {
      Wire.setClock((uint32_t)i2c_clk_hz);
      Serial.print(F("OK I2C clk=")); Serial.println(i2c_clk_hz);
    } else {
      Serial.println(F("Uso: #clk 100k | #clk 400k"));
    }
    return;
  }

  if (toks[0] == "#scan") {
    if (nt >= 2) {
      if (setClk(toks[1])) Wire.setClock((uint32_t)i2c_clk_hz);
      else Serial.println(F("Nota: clock no reconocido, usando el actual"));
    }
    i2cScan();
    return;
  }

  if (toks[0] == "#addr") {
    if (nt < 2) { Serial.println(F("Uso: #addr 0x36")); return; }
    uint32_t a;
    if (!parseUInt(toks[1], a) || a > 0x7F) { Serial.println(F("Addr invalida")); return; }
    i2c_addr_default = (uint8_t)a;
    Serial.print(F("OK addr_default=0x"));
    if (i2c_addr_default < 16) Serial.print('0');
    Serial.println(i2c_addr_default, HEX);
    return;
  }

  if (toks[0] == "#rd") {
    uint8_t addr = i2c_addr_default;
    uint32_t n = 0;

    if (nt == 2) {
      if (addr == 0x00) { Serial.println(F("Fija addr primero: #addr 0x.. o usa #rd 0x36 N")); return; }
      if (!parseUInt(toks[1], n)) { Serial.println(F("Uso: #rd N  o  #rd 0x36 N")); return; }
    } else if (nt >= 3) {
      uint32_t a;
      if (!parseUInt(toks[1], a) || a > 0x7F) { Serial.println(F("Addr invalida")); return; }
      addr = (uint8_t)a;
      if (!parseUInt(toks[2], n)) { Serial.println(F("Uso: #rd 0x36 N")); return; }
    } else {
      Serial.println(F("Uso: #rd N  o  #rd 0x36 N"));
      return;
    }

    if (n == 0 || n > 256) { Serial.println(F("N invalido (1..256)")); return; }
    i2cReadBytes(addr, (uint16_t)n);
    return;
  }

  if (toks[0] == "#wr") {
    if (nt < 3) { Serial.println(F("Uso: #wr 0x36 b1 b2 ...")); return; }
    uint32_t a;
    if (!parseUInt(toks[1], a) || a > 0x7F) { Serial.println(F("Addr invalida")); return; }
    uint8_t addr = (uint8_t)a;

    uint8_t buf[64];
    uint16_t bn = 0;
    for (int i = 2; i < nt && bn < sizeof(buf); i++) {
      uint32_t v;
      if (!parseUInt(toks[i], v) || v > 0xFF) { Serial.println(F("Byte invalido")); return; }
      buf[bn++] = (uint8_t)v;
    }
    if (bn == 0) { Serial.println(F("Sin datos")); return; }
    i2cWriteBytes(addr, buf, bn);
    return;
  }

  if (toks[0] == "#reg") {
    if (nt < 4) { Serial.println(F("Uso: #reg 0x36 0x04 N")); return; }
    uint32_t a, r, n;
    if (!parseUInt(toks[1], a) || a > 0x7F) { Serial.println(F("Addr invalida")); return; }
    if (!parseUInt(toks[2], r) || r > 0xFF) { Serial.println(F("Reg invalido")); return; }
    if (!parseUInt(toks[3], n) || n == 0 || n > 256) { Serial.println(F("N invalido (1..256)")); return; }
    i2cRegRead((uint8_t)a, (uint8_t)r, (uint16_t)n);
    return;
  }

  Serial.println(F("Comando no reconocido. Usa #h"));
}

// ------------------ Setup/Loop ------------------

// ------------------ Helpers UART (asistente) ------------------

static void uiPrintHeader() {
  Serial.println();
  Serial.println(F("=== SA01 Tool (Sketch) ==="));
}

static void uiPrintDiscoverMenu() {
  uiPrintHeader();
  Serial.println(F("1) Detectar / descubrir el dispositivo"));
  Serial.println(F("h) Ayuda"));
  Serial.println(F(""));
  Serial.print(F("> "));
}

static void uiPrintChooseUiMenu() {
  uiPrintHeader();
  if (det_link == DET_UART) Serial.println(F("Dispositivo detectado por UART."));
  else if (det_link == DET_I2C) {
    Serial.print(F("Dispositivo detectado por I2C (addr 0x"));
    if (det_i2c_addr < 16) Serial.print('0');
    Serial.print(det_i2c_addr, HEX);
    Serial.println(F(")."));
  } else {
    Serial.println(F("No se detecto el dispositivo."));
  }
  // Pausa de cortesía tras detectar (solo la primera vez)
  if (ui_choose_delay_pending && det_link != DET_NONE) {
    delay(3000);
  }
  ui_choose_delay_pending = false;
  Serial.println();
  Serial.println(F("Elige la interfaz:"));
  Serial.println(F("1) Asistente (recomendado)"));
  Serial.println(F("2) Modo tecnico (avanzado)"));
  Serial.println(F(""));
  Serial.print(F("> "));
}

static void uiPrintFriendlyMain() {
  if (ui_friendly_delay_pending) { delay(3000); ui_friendly_delay_pending = false; }
  uiPrintHeader();
  Serial.println(F("Que quieres hacer ahora?"));
  Serial.println(F("1) Cambiar modo de comunicacion"));
  Serial.println(F("2) Configurar alarma"));
  Serial.println(F("3) Obtener enlace del certificado (PRO)"));
  Serial.println(F("4) Probar lectura (viento y temperatura)"));
  Serial.println(F("0) Volver a detectar dispositivo"));
  Serial.println(F("9) Ir a modo tecnico"));
  Serial.println(F(""));
  Serial.print(F("> "));
}

static void uiDrainSerial2(bool forwardToUsb, uint32_t max_ms=30) {
  uint32_t t0 = millis();
  while ((millis() - t0) < max_ms) {
    while (Serial2.available()) {
      char c = (char)Serial2.read();
      if (forwardToUsb) Serial.write((uint8_t)c);
    }
    delay(1);
  }
}

static bool uartWaitForSubstring(const char *needle, uint32_t timeout_ms, String *captured=nullptr) {
  uint32_t t0 = millis();
  String buf;
  while ((millis() - t0) < timeout_ms) {
    while (Serial2.available()) {
      char c = (char)Serial2.read();
      buf += c;
      if (buf.length() > 800) buf.remove(0, buf.length() - 400);
      if (strstr(buf.c_str(), needle) != nullptr) {
        if (captured) *captured = buf;
        return true;
      }
    }
    delay(1);
  }
  if (captured) *captured = buf;
  return false;
}
// ---------------- CRC16 Modbus + parse trama UART SA01 ----------------
// CCC (3 digitos) = LSB del CRC16 Modbus calculado sobre el payload ASCII "AAAA WWWW TTTT" (14 bytes), init 0xFFFF.
static uint16_t crc16_modbus_bytes(const uint8_t *data, size_t len, uint16_t crc = 0xFFFF) {
  for (size_t pos = 0; pos < len; pos++) {
    crc ^= (uint16_t)data[pos];
    for (uint8_t i = 0; i < 8; i++) {
      if (crc & 0x0001) {
        crc >>= 1;
        crc ^= 0xA001;
      } else {
        crc >>= 1;
      }
    }
  }
  return crc;
}

// Lee una línea desde Serial2 hasta '\n' (incluye '\r' si llega). Devuelve true si se obtuvo línea.
static bool uartReadLine(String &line, uint32_t timeout_ms) {
  uint32_t t0 = millis();
  line = "";
  while ((millis() - t0) < timeout_ms) {
    while (Serial2.available()) {
      char c = (char)Serial2.read();
      // terminador de línea
      if (c == '\n') return true;
      line += c;
      // evita líneas enormes si hay ruido continuo
      if (line.length() > 160) line = "";
    }
    delay(1);
  }
  return false;
}

// Intenta parsear una trama tipo "AAAA WWWW TTTT CCC" dentro de una línea (puede estar mezclada con ruido).
static bool uartParseSa01Frame(const String &line_in,
                              uint16_t &analog,
                              uint16_t &wind10,
                              uint16_t &temp100,
                              uint8_t &ccc_rx,
                              uint8_t &ccc_calc,
                              bool &crc_ok) {
  String line = line_in;
  line.replace("\r", "");
  line.replace("\n", "");

  if (line.length() < 18) return false;

  int start = -1;
  // buscamos el patrón fijo de 18 chars (sin CRLF): dddd dddd dddd ddd
  for (int i = 0; i + 18 <= (int)line.length(); i++) {
    if (isDigit(line[i]) && isDigit(line[i+1]) && isDigit(line[i+2]) && isDigit(line[i+3]) &&
        line[i+4] == ' ' &&
        isDigit(line[i+5]) && isDigit(line[i+6]) && isDigit(line[i+7]) && isDigit(line[i+8]) &&
        line[i+9] == ' ' &&
        isDigit(line[i+10]) && isDigit(line[i+11]) && isDigit(line[i+12]) && isDigit(line[i+13]) &&
        line[i+14] == ' ' &&
        isDigit(line[i+15]) && isDigit(line[i+16]) && isDigit(line[i+17])) {
      start = i;
      break;
    }
  }
  if (start < 0) return false;

  String payload = line.substring(start, start + 14);  // "AAAA WWWW TTTT"
  String sA = line.substring(start, start + 4);
  String sW = line.substring(start + 5, start + 9);
  String sT = line.substring(start + 10, start + 14);
  String sC = line.substring(start + 15, start + 18);

  analog = (uint16_t)sA.toInt();
  wind10 = (uint16_t)sW.toInt();
  temp100 = (uint16_t)sT.toInt();
  ccc_rx = (uint8_t)sC.toInt();

  uint16_t crc16 = crc16_modbus_bytes((const uint8_t*)payload.c_str(), 14, 0xFFFF);
  ccc_calc = (uint8_t)(crc16 & 0xFF);
  crc_ok = (ccc_calc == ccc_rx);
  return true;
}

// Busca durante timeout_ms una trama SA01 válida (o al menos parseable) en Serial2.
static bool uartGetSa01Frame(uint32_t timeout_ms,
                            uint16_t &analog,
                            uint16_t &wind10,
                            uint16_t &temp100,
                            uint8_t &ccc_rx,
                            uint8_t &ccc_calc,
                            bool &crc_ok) {
  uint32_t t0 = millis();
  String line;

  while ((millis() - t0) < timeout_ms) {
    if (uartReadLine(line, 220)) {
      if (uartParseSa01Frame(line, analog, wind10, temp100, ccc_rx, ccc_calc, crc_ok)) {
        return true;
      }
    } else {
      // sin línea completa en esta ventana, seguimos
    }
  }
  return false;
}
// ----------------------------------------------------------------------


static void uartSendLine(const String &s) {
  Serial2.print(s);
  Serial2.print("\r\n");
}

static String makeLcmd(uint8_t idx, long value) {
  char b[24];
  // 9 digitos con signo (si fuese necesario). Ej: l00+000000002
  sprintf(b, "l%02u+%09ld", (unsigned)idx, (long)value);
  return String(b);
}

static bool uartWriteUserVar(uint8_t idx, long value) {
  String cmd = makeLcmd(idx, value);
  uartSendLine(cmd);
  // Esperamos eco del mismo comando
  bool ok = uartWaitForSubstring(cmd.c_str(), 1200);
  if (!ok) {
    Serial.println(F("No pude confirmar el cambio (sin eco)."));
    return false;
  }
  return true;
}

static bool uartCommit() {
  uartSendLine(F("COMMIT"));
  return uartWaitForSubstring("EEPROM USER SAVE DONE!", 2000);
}

static bool uartUnlock() {
  // SA01 puede fallar a la primera si hay "basura" previa en buffers o si el mapeo de pines
  // no ha quedado bien tras reset. Por eso:
  //  - hacemos un "kick" de Serial2 (pin-matrix)
  //  - drenamos RX
  //  - repetimos SA01 varias veces con timeout razonable (> 1s)
  uartKickInit();
  uiDrainSerial2(false, 60);

  const uint8_t tries = 3;
  const uint32_t per_try_ms = 1300; // > 1s (peor caso indicado)
  for (uint8_t i = 0; i < tries; i++) {
    uartSendLine(F("SA01"));
    if (uartWaitForSubstring("SA01", per_try_ms)) return true;
    uiDrainSerial2(false, 60);
    delay(80);
  }
  return false;
}


// Variante con timeout configurable (para detección tras reset)
bool uartUnlockTimeout(uint32_t ms) {
  // Presupuesto total ~ms, repartido en varios intentos para tolerar basura previa.
  // También aplica un "kick" por si Serial2 no queda mapeado tras reset.
  uartKickInit();
  uiDrainSerial2(false, 60);

  const uint8_t tries = 3;
  // Repartir el presupuesto total; asegurar un mínimo por intento.
  uint32_t per_try_ms = ms / tries;
  if (per_try_ms < 600) per_try_ms = 600;

  for (uint8_t i = 0; i < tries; i++) {
    uartSendLine(F("SA01"));
    if (uartWaitForSubstring("SA01", per_try_ms)) return true;
    uiDrainSerial2(false, 60);
    delay(80);
  }
  return false;
}

static bool uartInfoGet(String &serial, String &webpass, String &crc_calc, String &data_stat) {
  // INFO puede venir mezclado con tramas periódicas; por eso capturamos una ventana de tiempo
  // y parseamos por etiquetas. Mantener un buffer acotado evita crecer sin límite.
  uartSendLine(F("INFO"));

  String cap;
  uint32_t t0 = millis();
  const uint32_t CAP_MS = 2200;
  const size_t   CAP_MAX = 2048;

  while ((millis() - t0) < CAP_MS) {
    while (Serial2.available()) {
      char c = (char)Serial2.read();
      cap += c;
      if (cap.length() > CAP_MAX) {
        cap.remove(0, cap.length() - CAP_MAX);
      }
    }

    // Si ya tenemos todas las claves, podemos salir antes
    if (cap.indexOf("Serial:") >= 0 &&
        cap.indexOf("WEBPass:") >= 0 &&
        cap.indexOf("CRC_CALC:") >= 0 &&
        cap.indexOf("DATA_STAT:") >= 0) {
      // intentamos asegurarnos de que al menos llegó el fin de línea de DATA_STAT
      int p = cap.indexOf("DATA_STAT:");
      if (p >= 0) {
        int eol = cap.indexOf('\n', p);
        if (eol >= 0) break;
      }
    }
    delay(2);
  }

  auto grabToken = [&](const char* key)->String {
    int p = cap.indexOf(key);
    if (p < 0) return String();
    p += (int)strlen(key);
    // saltar espacios
    while (p < (int)cap.length() && (cap[p] == ' ' || cap[p] == '\t')) p++;
    int e = cap.indexOf('\n', p);
    if (e < 0) e = (int)cap.length();
    String out = cap.substring(p, e);
    out.replace("\r", "");
    out.trim();
    return out;
  };

  serial    = grabToken("Serial:");
  webpass   = grabToken("WEBPass:");
  String crcLine = grabToken("CRC_CALC:");
  data_stat = grabToken("DATA_STAT:");

  // Normalizar CRC_CALC: aceptar "0x1234" y quedarnos con 4 hex sin 0x
  crc_calc = crcLine;
  crc_calc.replace("0x", "");
  crc_calc.replace("0X", "");
  crc_calc.trim();

  return (serial.length() > 0 && webpass.length() > 0 && crc_calc.length() > 0);
}


// ------------------ Descubrimiento ------------------

static void uiDetectDevice() {
  Serial.println();
  Serial.println(F("Buscando el dispositivo..."));

  // Asegurar UART activo para intentar eco
  if (mode == MODE_I2C_TOOL) exitI2CToUart(false);

  // Re-enganche robusto antes de detectar (evita tener que hacer #swap dos veces)
  uartKickInit();
  // Warm-up corto: tras reset, a veces el pin-matrix/UART tarda en estabilizar
  delay(350);
  uiDrainSerial2(false, 80);

  det_link = DET_NONE;

  // 1) Intento pasivo: llegan datos por UART?
  uint32_t t0 = millis();
  while (millis() - t0 < 3500) {
    if (Serial2.available()) {
      det_link = DET_UART;
      break;
    }
    delay(1);
  }

  // 2) Intento activo: pedir eco SA01
  if (det_link == DET_NONE) {
    Serial.println(F("Intentando por UART..."));
    if (uartUnlockTimeout(3500)) {
      det_link = DET_UART;
    }
  } else {
    // drenar sin imprimir
    uiDrainSerial2(false, 50);
  }

  if (det_link == DET_UART) {
    Serial.println(F("OK: detectado por UART."));
    // Mantener en UART
    det_i2c_addr = i2c_addr_default;
    return;
  }

  // 2b) Reintento automático por UART (útil justo tras reset)
  if (det_link == DET_NONE) {
    Serial.println(F("Reintentando por UART..."));
    uartKickInit();

    // Pasivo: esperar datos
    uint32_t t1 = millis();
    while (millis() - t1 < 3500) {
      if (Serial2.available()) {
        uiDrainSerial2(false, 50);
        det_link = DET_UART;
        break;
      }
      delay(1);
    }

    // Activo: pedir eco SA01 con timeout más amplio
    if (det_link == DET_NONE) {
      if (uartUnlockTimeout(3500)) {
        det_link = DET_UART;
      }
    } else {
      uiDrainSerial2(false, 50);
    }

    if (det_link == DET_UART) {
      Serial.println(F("OK: detectado por UART."));
      det_i2c_addr = i2c_addr_default;
      return;
    }
  }


  // 3) Intento I2C: parar UART y probar bus
  Serial.println(F("Intentando por I2C..."));
  enterI2C(false, false);

  // Probar addr por defecto primero
  float w, t;
  uint16_t rw, rt; uint8_t got;
  if (i2cQuickReadWindTemp(i2c_addr_default, w, t, rw, rt, got)) {
    det_link = DET_I2C;
    det_i2c_addr = i2c_addr_default;
    Serial.print(F("OK: detectado por I2C en 0x"));
    if (det_i2c_addr < 16) Serial.print('0');
    Serial.println(det_i2c_addr, HEX);
    return;
  }

  // Scan + validacion (4 bytes)
  uint8_t found = 0;
  for (uint8_t a = 0x08; a <= 0x77; a++) {
    Wire.beginTransmission(a);
    if (Wire.endTransmission() != 0) continue;

    if (i2cQuickReadWindTemp(a, w, t, rw, rt, got)) {
      det_link = DET_I2C;
      det_i2c_addr = a;
      found = 1;
      break;
    }
  }

  if (found) {
    Serial.print(F("OK: detectado por I2C en 0x"));
    if (det_i2c_addr < 16) Serial.print('0');
    Serial.println(det_i2c_addr, HEX);
    return;
  }

  det_link = DET_NONE;
  Serial.println();
  Serial.println(F("No se detecto el dispositivo."));
  Serial.println(F("Si el anemometro esta en modo I2C o sin UART, puede ser necesario:"));
  Serial.println(F(" - Desconectar alimentacion"));
  Serial.println(F(" - Puentear pines 1 y 5 ANTES de alimentar"));
  Serial.println(F(" - Alimentar, y luego retirar el puente"));
}

// ------------------ Loop UI (menu + asistente) ------------------

static void uiGoDiscoverMenu() {
  menu_state = MS_NEED_DISCOVER;
  uiPrintDiscoverMenu();
}

static void uiGoChooseUiMenu() {
  menu_state = MS_CHOOSE_UI;
  ui_choose_delay_pending = true;
  uiPrintChooseUiMenu();
}

static void uiGoFriendlyMain() {
  menu_state = MS_FRIENDLY_MAIN;
  uiPrintFriendlyMain();
}

static void uiGoFriendlyMainDelay() {
  ui_friendly_delay_pending = true;
  uiGoFriendlyMain();
}

static bool uiParseLong(const String &s, long &out) {
  char *endp=nullptr;
  long v = strtol(s.c_str(), &endp, 10);
  if (endp == s.c_str()) return false;
  out = v;
  return true;
}

static void uiShowSummaryModeChoice() {
  Serial.println();
  Serial.println(F("Resumen:"));
  if (wiz_mode_choice == 0) Serial.println(F(" - Modo: UART (recomendado)"));
  else if (wiz_mode_choice == 3) Serial.println(F(" - Modo: I2C"));
  else if (wiz_mode_choice == 1) Serial.println(F(" - Modo: Alarma por patilla (se puede configurar por UART)"));
  else if (wiz_mode_choice == 2) Serial.println(F(" - Modo: Alarma por patilla (sin configuracion por UART)"));
}

static void uiApplyModeChoiceAndCommit() {
  // Requiere UART
  if (mode == MODE_I2C_TOOL) exitI2CToUart(false);

  if (!uartUnlock()) {
    Serial.println(F("No pude desbloquear por UART. Revisa conexion o modo del dispositivo."));
    return;
  }

  // Map wizard choice to l00
  long l00 = 0;
  if (wiz_mode_choice == 0) l00 = 0;
  else if (wiz_mode_choice == 1) l00 = 1;
  else if (wiz_mode_choice == 2) l00 = 2;
  else if (wiz_mode_choice == 3) l00 = 3;

  Serial.println(F("Aplicando cambio..."));
  if (!uartWriteUserVar(0, l00)) return;

  if (!uartCommit()) {
    Serial.println(F("No se pudo guardar (COMMIT)."));
    return;
  }

  Serial.println(F("Listo. Cambios guardados."));
  if (l00 == 2 || l00 == 3) {
    Serial.println(F("Nota: si necesitas reconfigurar mas adelante, puede requerir puente 1-5 antes de alimentar."));
  }
}

static void uiApplyAlarmAndCommit() {
  if (mode == MODE_I2C_TOOL) exitI2CToUart(false);

  if (!uartUnlock()) {
    Serial.println(F("No pude desbloquear por UART. Revisa conexion o modo del dispositivo."));
    return;
  }

  Serial.println(F("Aplicando configuracion de alarma..."));

  if (!uartWriteUserVar(1, wiz_l01)) return;
  if (!uartWriteUserVar(2, wiz_l02)) return;
  if (!uartWriteUserVar(3, wiz_l03)) return;
  if (!uartWriteUserVar(4, wiz_l04)) return;
  if (!uartWriteUserVar(5, wiz_l05)) return;

  if (!uartCommit()) {
    Serial.println(F("No se pudo guardar (COMMIT)."));
    return;
  }

  Serial.println(F("Listo. Alarma guardada."));
}

static void uiFriendlyHandleLine(const String &cmd) {
  if (cmd.length() == 0) { Serial.print(F("> ")); return; }

  // comandos universales del asistente
  if (cmd.equalsIgnoreCase("x")) {
    uiGoFriendlyMainDelay();
    return;
  }

  switch (menu_state) {
    case MS_NEED_DISCOVER: {
      if (cmd == "1") {
        uiDetectDevice();
        if (det_link != DET_NONE) uiGoChooseUiMenu();
        else uiGoDiscoverMenu();
      } else if (cmd.equalsIgnoreCase("h")) {
        Serial.println();
        Serial.println(F("Este asistente te ayuda a detectar y configurar el SA01."));
        Serial.println(F("1 = detectar dispositivo"));
        uiGoDiscoverMenu();
      } else {
        Serial.println(F("Opcion no valida."));
        uiGoDiscoverMenu();
      }
    } break;

    case MS_CHOOSE_UI: {
      if (cmd == "1") {
        ui_mode = UI_FRIENDLY;
        uiGoFriendlyMain();
      } else if (cmd == "2") {
        ui_mode = UI_TECH;
        Serial.println(F("\nEntrando a modo tecnico. Usa #h para ayuda. Para volver al menu: #ui"));
      } else {
        Serial.println(F("Opcion no valida."));
        uiPrintChooseUiMenu();
      }
    } break;

    case MS_FRIENDLY_MAIN: {
      if (cmd == "1") {
        Serial.println();
        Serial.println(F("Como quieres que funcione el dispositivo?"));
        Serial.println(F("1) UART (recomendado)"));
        Serial.println(F("2) Alarma por patilla (se puede configurar por UART)"));
        Serial.println(F("3) Alarma por patilla (sin configuracion por UART)"));
        Serial.println(F("4) I2C"));
        Serial.println();
        Serial.print(F("> "));
        menu_state = MS_FRIENDLY_SET_MODE_PICK;
      } else if (cmd == "2") {
        Serial.println();
        Serial.println(F("Vamos a configurar la alarma."));
        Serial.println(F("A partir de cuantos km/h quieres que se active? (entero)"));
        Serial.print(F("> "));
        menu_state = MS_FRIENDLY_ALARM_WIND;
      } else if (cmd == "3") {
        Serial.println();
        if (mode == MODE_I2C_TOOL) exitI2CToUart(false);
        if (!uartUnlock()) {
          Serial.println(F("No puedo leer INFO por UART. Si el dispositivo esta en I2C, puede requerir puente 1-5 para volver a UART."));
          uiGoFriendlyMainDelay();
          return;
        }
        String sn, wp, crc, st;
        if (!uartInfoGet(sn, wp, crc, st)) {
          Serial.println(F("No pude obtener la informacion del certificado."));
          uiGoFriendlyMainDelay();
          return;
        }
        Serial.println(F("Listo. Enlace para el certificado:"));
        Serial.print(F("https://server-doc.com/claim/quick?serial="));
        Serial.print(sn);
        Serial.print(F("&webpass="));
        Serial.println(wp);

        if (st.length()) {
          Serial.print(F("Estado de integridad: "));
          Serial.println(st);
          if (st.equalsIgnoreCase("CORRUPT")) {
            Serial.println(F("Nota: CORRUPT significa que la configuracion actual no coincide con la original (CRC diferente)."));
          }
        }

        Serial.println(F("Nota: el certificado completo esta disponible para versiones PRO; en otras versiones la web puede mostrar info limitada."));
        uiGoFriendlyMainDelay();
      } else if (cmd == "4") {
        Serial.println();
        if (det_link == DET_I2C) {
          enterI2C(false, false);
          i2cQuickReadAndPrint(det_i2c_addr);
        } else if (det_link == DET_UART) {
          Serial.println(F("Leyendo una trama por UART..."));
          uint16_t a = 0, w10 = 0, t100 = 0;
          uint8_t ccc_rx = 0, ccc_calc = 0;
          bool crc_ok = false;

          if (uartGetSa01Frame(2200, a, w10, t100, ccc_rx, ccc_calc, crc_ok)) {
            float wind_kmh = ((float)w10) / 10.0f;
            float temp_c = ((float)t100) / 100.0f - 40.0f;

            Serial.print(F("Viento: ")); Serial.print(wind_kmh, 1); Serial.println(F(" km/h"));
            Serial.print(F("Temp:   ")); Serial.print(temp_c, 2);    Serial.println(F(" C"));
            Serial.print(F("CRC:    "));
            if (crc_ok) Serial.println(F("OK"));
            else {
              Serial.print(F("ERROR (rx="));
              Serial.print((unsigned int)ccc_rx);
              Serial.print(F(" calc="));
              Serial.print((unsigned int)ccc_calc);
              Serial.println(F(")"));
            }
            // Para diagnóstico adicional (no técnico, pero útil):
            Serial.print(F("RAW:    A=")); Serial.print(a);
            Serial.print(F(" W10=")); Serial.print(w10);
            Serial.print(F(" T100=")); Serial.println(t100);
          } else {
            Serial.println(F("No se recibieron tramas de medida por UART."));
            Serial.println(F("Posibles causas:"));
            Serial.println(F(" - El dispositivo no esta emitiendo datos por TX (por ejemplo, TX usado como alarma)."));
            Serial.println(F(" - Cableado/baudios incorrectos."));
            Serial.println(F(" - El dispositivo esta en modo I2C (sin UART)."));
          }
        } else {
          Serial.println(F("Primero detecta el dispositivo (opcion 0)."));
        }
        uiGoFriendlyMainDelay();
      } else if (cmd == "0") {
        uiGoDiscoverMenu();
      } else if (cmd == "9") {
        ui_mode = UI_TECH;
        Serial.println(F("\nEntrando a modo tecnico. Usa #h para ayuda. Para volver al menu: #ui"));
      } else {
        Serial.println(F("Opcion no valida."));
        uiGoFriendlyMainDelay();
      }
    } break;

    case MS_FRIENDLY_SET_MODE_PICK: {
      if (cmd == "1") wiz_mode_choice = 0;
      else if (cmd == "2") wiz_mode_choice = 1;
      else if (cmd == "3") wiz_mode_choice = 2;
      else if (cmd == "4") wiz_mode_choice = 3;
      else {
        Serial.println(F("Opcion no valida."));
        Serial.print(F("> "));
        return;
      }

      uiShowSummaryModeChoice();
      if (wiz_mode_choice == 2 || wiz_mode_choice == 3) {
        Serial.println(F("Aviso: para reconfigurar mas adelante puede ser necesario un puente (1-5) antes de alimentar."));
      }
      Serial.println();
      Serial.println(F("Confirmas este cambio? (S/N)"));
      Serial.print(F("> "));
      menu_state = MS_FRIENDLY_SET_MODE_CONFIRM;
    } break;

    case MS_FRIENDLY_SET_MODE_CONFIRM: {
      if (cmd.equalsIgnoreCase("s") || cmd.equalsIgnoreCase("si") || cmd.equalsIgnoreCase("y") || cmd.equalsIgnoreCase("yes")) {
        uiApplyModeChoiceAndCommit();
        uiGoFriendlyMainDelay();
      } else {
        Serial.println(F("Cancelado."));
        uiGoFriendlyMainDelay();
      }
    } break;

    case MS_FRIENDLY_ALARM_WIND: {
      long v;
      if (!uiParseLong(cmd, v) || v < 0 || v > 32768) {
        Serial.println(F("Valor no valido. Ingresa un entero 0..32768."));
        Serial.print(F("> "));
        return;
      }
      wiz_l01 = v;

      Serial.println(F("Cuantos segundos debe mantenerse para activarse? (entero)"));
      Serial.print(F("> "));
      menu_state = MS_FRIENDLY_ALARM_TIME_ON;
    } break;

    case MS_FRIENDLY_ALARM_TIME_ON: {
      long v;
      if (!uiParseLong(cmd, v) || v < 0 || v > 32768) {
        Serial.println(F("Valor no valido. Ingresa un entero 0..32768."));
        Serial.print(F("> "));
        return;
      }
      wiz_l02 = v;

      Serial.println(F("Despues de activarse, cuantos segundos quieres que espere antes de poder activarse otra vez?"));
      Serial.print(F("> "));
      menu_state = MS_FRIENDLY_ALARM_HOLDOFF;
    } break;

    case MS_FRIENDLY_ALARM_HOLDOFF: {
      long v;
      if (!uiParseLong(cmd, v) || v < 0 || v > 32768) {
        Serial.println(F("Valor no valido. Ingresa un entero 0..32768."));
        Serial.print(F("> "));
        return;
      }
      wiz_l03 = v;

      Serial.println(F("Cuantos segundos dura el pulso de alarma? (1..32768)"));
      Serial.print(F("> "));
      menu_state = MS_FRIENDLY_ALARM_PULSE;
    } break;

    case MS_FRIENDLY_ALARM_PULSE: {
      long v;
      if (!uiParseLong(cmd, v) || v < 1 || v > 32768) {
        Serial.println(F("Valor no valido. Ingresa un entero 1..32768."));
        Serial.print(F("> "));
        return;
      }
      wiz_l04 = v;

      Serial.println(F("En reposo (sin alarma), que nivel quieres en la salida?"));
      Serial.println(F("1) 0V"));
      Serial.println(F("2) 3.3V"));
      Serial.print(F("> "));
      menu_state = MS_FRIENDLY_ALARM_IDLELEVEL;
    } break;

    case MS_FRIENDLY_ALARM_IDLELEVEL: {
      if (cmd == "1") wiz_l05 = 0;
      else if (cmd == "2") wiz_l05 = 1;
      else {
        Serial.println(F("Opcion no valida."));
        Serial.print(F("> "));
        return;
      }

      Serial.println();
      Serial.println(F("Resumen de alarma:"));
      Serial.print(F(" - Se activa a partir de: ")); Serial.print(wiz_l01); Serial.println(F(" km/h"));
      Serial.print(F(" - Confirmacion: ")); Serial.print(wiz_l02); Serial.println(F(" s"));
      Serial.print(F(" - Anti-reentrada: ")); Serial.print(wiz_l03); Serial.println(F(" s"));
      Serial.print(F(" - Pulso: ")); Serial.print(wiz_l04); Serial.println(F(" s"));
      Serial.print(F(" - Reposo salida: ")); Serial.println((wiz_l05==0)?F("0V"):F("3.3V"));

      Serial.println();
      Serial.println(F("Confirmas guardar esta alarma? (S/N)"));
      Serial.print(F("> "));
      menu_state = MS_FRIENDLY_ALARM_CONFIRM;
    } break;

    case MS_FRIENDLY_ALARM_CONFIRM: {
      if (cmd.equalsIgnoreCase("s") || cmd.equalsIgnoreCase("si") || cmd.equalsIgnoreCase("y") || cmd.equalsIgnoreCase("yes")) {
        uiApplyAlarmAndCommit();
        uiGoFriendlyMainDelay();
      } else {
        Serial.println(F("Cancelado."));
        uiGoFriendlyMainDelay();
      }
    } break;

    default:
      uiGoDiscoverMenu();
      break;
  }
}

static void loopUi(uint32_t now) {
  // En UI no volcamos UART a pantalla por defecto (evita ruido).
  // Solo procesamos entradas del usuario.
  while (Serial.available()) {
    char c = (char)Serial.read();
    if (c == '\r') continue;
    if (c == '\n') {
      String cmd = line;
      cmd.trim();
      if (cmd.length()) {
        // Eco local para que se vea la opción ingresada a continuación del prompt "> "
        Serial.println(cmd);
      } else {
        Serial.println();
      }
      uiFriendlyHandleLine(cmd);
      line = "";
    } else {
      line += c;
      if (line.length() > 220) line = "";
    }
  }
}

void setup() {
  Serial.begin(115200);
  delay(200);

  // Inicial: modo puente UART
  PIN_A = 13;
  PIN_B = 25;
  uart_rx = PIN_A;
  uart_tx = PIN_B;
// Inicial: UART activo (pero el sketch arranca en MENU, sin reenviar nada)
i2c_armed = false;
startUart();
  uartKickInit();
// --- UI: arrancar en menu (sin tecnicismos) ---
ui_mode = UI_MENU;
menu_state = MS_NEED_DISCOVER;
uiGoDiscoverMenu();
}


void loop() {
  uint32_t now = millis();

  if (ui_mode == UI_TECH) {
    loopTech(now);
    return;
  }

  // MENU o ASISTENTE
  loopUi(now);
}


static void loopTech(uint32_t now) {
  // Auto-lectura rápida (solo en I2C)
  if (mode == MODE_I2C_TOOL) {
    if (auto_interval_ms > 0 && (int32_t)(now - auto_next_ms) >= 0) {
      // Mientras AUTO está activo, consideramos esto actividad para NO auto-salir a UART
      // aunque haya fallos (queremos debug I2C continuo).
      last_i2c_activity_ms = millis();
      i2c_deadline_ms = last_i2c_activity_ms + arm_seconds * 1000UL;

      i2cQuickReadAndPrint(i2c_addr_default);
      auto_next_ms = millis() + auto_interval_ms;

      // IMPORTANTE: refrescar 'now' para que el cálculo de dt no haga underflow
      // (i2cQuickReadAndPrint actualiza last_i2c_activity_ms con millis() más nuevo).
      now = millis();
    }
  }

  // Auto-switch por inactividad (si está habilitado)
  if (i2c_armed) {
    if (mode == MODE_I2C_TOOL) {
      // Si está el AUTO de lectura activo, NO hacemos auto-switch a UART.
      if (auto_interval_ms > 0) {
        last_i2c_activity_ms = now;
        i2c_deadline_ms = now + arm_seconds * 1000UL;
      } else {
        uint32_t dt = now - last_i2c_activity_ms;
        if (dt >= arm_seconds * 1000UL) {
          Serial.println(F("\nAUTO-SWITCH: I2C idle -> UART"));
          exitI2CToUart();
        } else {
          i2c_deadline_ms = last_i2c_activity_ms + arm_seconds * 1000UL;
        }
      }
    } else {
      // Timeout UART -> I2C
      if (!lock_uart) {
        uint32_t dt = now - last_uart_activity_ms;
        if (dt >= arm_seconds * 1000UL) {
          Serial.println(F("\nAUTO-SWITCH: UART idle -> I2C"));
          enterI2C(true);
        } else {
          i2c_deadline_ms = last_uart_activity_ms + arm_seconds * 1000UL;
        }
      } else {
        i2c_deadline_ms = now + arm_seconds * 1000UL;
      }
    }
  }

  // UART -> USB (solo en modo puente)
  if (mode == MODE_UART_BRIDGE) {
    while (Serial2.available()) {
      int c = Serial2.read();
      if (c >= 0) {
        last_uart_activity_ms = millis();
        i2c_deadline_ms = last_uart_activity_ms + arm_seconds * 1000UL;
        Serial.write((uint8_t)c);
      }
    }
  }

  // USB -> (comando local o reenvío)
  while (Serial.available()) {
    char c = (char)Serial.read();
    if (c == '\r') continue;

    if (c == '\n') {
      String cmd = line;
      cmd.trim();

      if (cmd.length() > 0) {
        if (cmd[0] == '#') {
          handleLocalCommand(cmd);
        } else {
          // Si estamos en UART bridge, reenviamos al anemómetro
          if (mode == MODE_UART_BRIDGE) {
            Serial2.print(cmd);
            Serial2.print('\n');
          } else {
            // Atajos rapidos: SOLO en modo I2C (para no interferir con UART)
            if (cmd.length() == 1) {
              char k = cmd[0];
              if (k >= '1' && k <= '7') {
                uint32_t now = millis();
                if (k == '1') {
                  auto_interval_ms = 0;
                  auto_next_ms = 0;
                  i2cQuickReadAndPrint(i2c_addr_default);
                } else if (k == '2') {
                  // toggle 1s
                  if (auto_interval_ms == 1000) auto_interval_ms = 0; else auto_interval_ms = 1000;
                  auto_next_ms = now;
                  Serial.print(F("AUTO interval ms=")); Serial.println(auto_interval_ms);
                } else if (k == '3') {
                  // toggle 100ms
                  if (auto_interval_ms == 100) auto_interval_ms = 0; else auto_interval_ms = 100;
                  auto_next_ms = now;
                  Serial.print(F("AUTO interval ms=")); Serial.println(auto_interval_ms);
                } else {
                  // 4..7 clocks
                  if (k == '4' && hz != 400000) hz = 400000;
                  else if (k == '5' && hz != 500000) hz = 500000;
                  else if (k == '6' && hz != 600000) hz = 600000;
                  else if (k == '7' && hz != 650000) hz = 650000;
                  else if (k >= '4' && k <= '7') hz = 100000;
                  i2c_clk_hz = hz;
                  Wire.setClock((uint32_t)i2c_clk_hz);
                  Serial.print(F("OK I2C clk=")); Serial.println(i2c_clk_hz);
                }
                // no mostramos warning
                line = "";
                return;
              }
            }
            Serial.println(F("Estas en modo I2C. Usa comandos '#...' o '#panic' para volver."));
          }
        }
      }
      line = "";
    } else {
      line += c;
      if (line.length() > 220) {
        // Evitar líneas gigantes; en UART mode lo volcamos
        if (mode == MODE_UART_BRIDGE) {
          Serial2.print(line);
        }
        line = "";
      }
    }
  }
}
